/**
 * Collection of types used for route paths to /otp/router/{ignoredRouterId}/plan
 */

import type { OtpDate, OtpLocal, OtpMode, OtpOptimze, OtpTime, OtpModeCombination } from "./base"

/**
 * Either __latitude, longitude__ pair in degrees or a __Vertex label__. 
 * 
 * @example
 * 40.714476,-74.005966 or mtanyctsubway_A27_S.
 */
export type QueryParamPlace = string

/**
 * An ordered list of intermediate locations to be visited (see the fromPlace for format). Parameter can be specified multiple times.
 * 
 * @example
 * 40.714476,-74.005966 or mtanyctsubway_A27_S.
 */
type QueryParamPlaces = QueryParamPlace
/**
 * @see http://dev.opentripplanner.org/apidoc/2.3.0/resource_PlannerResource.html
 */
export interface PlanQueryParams {
  time: OtpTime,
  /**
   * The date that the trip should depart (or arrive, for requests where arriveBy is true).
  */
  date: OtpDate,
  /**
   * The start location -- either latitude, longitude pair in degrees or a Vertex label. For example, 40.714476,-74.005966 or mtanyctsubway_A27_S
  */
  fromPlace: QueryParamPlace,
  /**
   * The end location -- either latitude, longitude pair in degrees or a Vertex label. For example, 40.714476,-74.005966 or mtanyctsubway_A27_S
  */
  toPlace: QueryParamPlace,

  locale: OtpLocal,

  /**
   * The set of modes that a user is willing to use, with qualifiers stating whether vehicles should be parked, rented, etc.
   * The possible values of the comma-separated list are:
   */
  mode: OtpMode | OtpModeCombination,

  /**
   * Flag signifying date + time
   */
  arriveBy: boolean,

  /**
   * The number of seconds to add after alighting a transit leg. It is recommended to use the alightTimes in the router-config.json to set this for each mode.
   * Unit is seconds. Default value is 0.
   * 
   * *Number represented as an __integer__*
   */
  alightSlack?: number,
  allowKeepingRentedBicycleAtDestination?: boolean,
  allowedVehicleRentalNetworks?: unknown,
  bannedAgencies?: unknown,
  bannedRoutes?: unknown,
  bannedStops?: unknown,
  bannedStopsHard?: unknown,
  bannedTrips?: unknown,
  bannedVehicleParkingTags?: unknown,
  bannedVehicleRentalNetworks?: unknown,
  batch?: boolean,
  /**
   * *Cost represented as an __integer__ number*
   */
  bikeBoardCost?: number,
  /**
   * *Cost represented as an __integer__ number*
   */
  bikeParkCost?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  bikeParkTime?: number,

  /**
   * *Time represented as a __floating point__ number*
   */
  bikeReluctance?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  bikeSpeed?: number,

  /**
   * *Time represented as an __integer__ number*
   */
  bikeSwitchCost?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  bikeSwitchTime?: number,

  /**
   * *Time represented as a __floating point__ number*
   */
  bikeWalkingReluctance?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  bikeWalkingSpeed?: number,

  /**
   * The number of seconds to add before boarding a transit leg. It is recommended to use the boardTimes in the router-config.json to set this for each mode.
   * Unit is __seconds__. Default value is _0_.
   * 
   * *Time represented as an __integer__ number*
   */
  boardSlack?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  carParkCost?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  carParkTime?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  carReluctance?: number,
  /**
   * When subtracting initial wait time, do not subtract more than this value, to prevent overly optimistic trips. Reasoning is that it is reasonable to delay a trip start 15 minutes to make a better trip, but that it is not reasonable to delay a trip start 15 hours; if that is to be done, the time needs to be included in the trip time. This number depends on the transit system; for transit systems where trips are planned around the vehicles, this number can be much higher. For instance, it's perfectly reasonable to delay one's trip 12 hours if one is taking a cross-country Amtrak train from Emeryville to Chicago. Has no effect in stock OTP, only in Analyst.
   * A value of 0 means that initial wait time will not be subtracted out (will be clamped to 0). A value of -1 (the default) means that clamping is disabled, so any amount of initial wait time will be subtracted out.
   * 
   * *Time represented as a __floating point__ long number*
   */
  clampInitialWait?: number,

  /**
   * This takes a RouteRequest as JSON and uses it as the default before applying other parameters. This is intended for debugging only! The RouteRequest is an internal OTP class and will change without notice. The JSON is the same as the one in the router-config for the "routingDefaults" parameter.
   */
  config?: unknown,
  debugItineraryFilter?: boolean,
  debugRaptorPath?: unknown,
  debugRaptorStops?: unknown,
  disableAlertFiltering?: boolean,
  disableRemainingWeightHeuristic?: boolean,
  geoidElevation?: boolean,
  /**
   * *Time represented as a __floating point__ number*
   */
  groupSimilarityKeepOne?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  groupSimilarityKeepThree?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  groupedOtherThanSameLegsMaxCostMultiplier?: number,
  ignoreRealtimeUpdates?: boolean,
  /**
   * An ordered list of intermediate locations to be visited (see the fromPlace for format). Parameter can be specified multiple times.
   */
  intermediatePlaces?: QueryParamPlaces,
  /**
   * *Time represented as a __floating point__ number*
   */
  keepingRentedBicycleAtDestinationCost?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  maxAdditionalTransfers?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  maxHours?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  maxPreTransitTime?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  maxTransfers?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  minTransferTime?: number,
  nonTransitGeneralizedCostLimitFunction?: unknown,
  /**
   * *Time represented as an __integer__ number*
   */
  nonpreferredTransferPenalty?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  numItineraries?: number,
  optimize?: OtpOptimze,
  /**
   * *Time represented as an __integer__ number*
   */
  otherThanPreferredRoutesPenalty?: number,
  pageCursor?: unknown,
  /**
   * The comma-separated list of preferred agencies
   */
  preferredAgencies?: string,
  /**
   * The list of preferred routes. The format is agency_[routename][_routeid], so TriMet_100 (100 is route short name) or Trimet__42 (two underscores, 42 is the route internal ID).
   */
  preferredRoutes?: string,
  /**
   * *Time represented as a __floating point__ number*
   */
  relaxTransitSearchGeneralizedCostAtDestination?: number,
  requiredVehicleParkingTags?: unknown,
  /**
   * THIS PARAMETER IS NO LONGER IN USE.
   * @deprecated
   */
  reverseOptimizeOnTheFly?: boolean,
  /**
   * The length of the search-window in __seconds__. This parameter is optional.
   * The search-window is defined as the duration between the earliest-departure-time(EDT) and the latest-departure-time(LDT). OTP will search for all itineraries in this departure window. If arriveBy=true the dateTime parameter is the latest-arrival-time, so OTP will dynamically calculate the EDT. Using a short search-window is faster than using a longer one, but the search duration is not linear. Using a \"too\" short search-window will waste resources server side, while using a search-window that is too long will be slow.
   * 
   * *Time represented as an __integer__ number*
   */
  searchWindow?: number,
  showIntermediateStops?: boolean,
  startTransitStopId?: unknown,
  startTransitTripId?: unknown,
  timetableView?: boolean,
  /**
   * *Time represented as an __integer__ number*
   */
  transferPenalty?: number,
  transitGeneralizedCostLimitFunction?: unknown,
  /**
   * *Time represented as a __floating point__ number*
   */
  transitGeneralizedCostLimitIntervalRelaxFactor?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  triangleSafetyFactor?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  triangleSlopeFactor?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  triangleTimeFactor?: number,
  unpreferredAgencies?: unknown,
  unpreferredRoutes?: unknown,
  useRequestedDateTimeInMaxHours?: boolean,
  useVehicleParkingAvailabilityInformation?: boolean,
  /**
   * How much worse is waiting for a transit vehicle than being on a transit vehicle, as a multiplier. The default value treats wait and on-vehicle time as the same.
   * It may be tempting to set this higher than walkReluctance (as studies often find this kind of preferences among riders) but the planner will take this literally and walk down a transit line to avoid waiting at a stop. This used to be set less than 1 (0.95) which would make waiting offboard preferable to waiting onboard in an interlined trip. That is also undesirable.
   * 
   * If we only tried the shortest possible transfer at each stop to neighboring stop patterns, this problem could disappear.
   * 
   * *Time represented as a __floating point__ number*
   */
  waitReluctance?: number,
  /**
   * *Time represented as an __integer__ number*
   */
  walkBoardCost?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  walkReluctance?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  walkSafetyFactor?: number,
  /**
   * *Time represented as a __floating point__ number*
   */
  walkSpeed?: number,
  wheelchair?: boolean,
  whiteListedAgencies?: unknown,
  whiteListedRoutes?: unknown,
}
